import warnings
from collections.abc import Callable, Sequence
from dataclasses import dataclass
# from typing import Any, Generic, Literal, TypeVar, cast
from typing import Any, Generic, Literal, TypeVar, cast, Type, Union, Dict

import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
from sklearn import neighbors  

from tianshou.data import (
    Batch,
    ReplayBuffer,
    to_torch,
    to_torch_as,
)

from tianshou.policy import BasePolicy
from tianshou.utils import RunningMeanStd
from tianshou.utils.net.continuous import ActorProb
from tianshou.utils.net.discrete import Actor

from Policy.Reward.fpg_reward import FPGReward
# Dimension Naming Convention
# B - Batch Size
# A - Action
# D - Dist input (usually 2, loc and scale)
# H - Dimension of hidden, can be None



class FPGPolicy(BasePolicy):
    """Implementation of FPG Algorithm.

    :param actor: the actor network following the rules:
        If `self.action_type == "discrete"`: (`s_B` ->`action_values_BA`).
        If `self.action_type == "continuous"`: (`s_B` -> `dist_input_BD`).
    :param optim: optimizer for actor network.
    :param dist_fn: distribution class for computing the action.
        Maps model_output -> distribution. Typically a Gaussian distribution
        taking `model_output=mean,std` as input for continuous action spaces,
        or a categorical distribution taking `model_output=logits`
        for discrete action spaces. Note that as user, you are responsible
        for ensuring that the distribution is compatible with the action space.
    :param action_space: env's action space.
    :param eps_clip: :math:`\epsilon` in :math:`L_{CLIP}` in the original
        paper.
    :param dual_clip: a parameter c mentioned in arXiv:1912.09729 Equ. 5,
        where c > 1 is a constant indicating the lower bound. Set to None
        to disable dual-clip PPO.
    :param ent_coef: weight for entropy loss.
    :param max_grad_norm: clipping gradients in back propagation.
    :param max_batchsize: the maximum size of the batch when computing GAE.
    :param discount_factor: in [0, 1].
    :param reward_normalization: if True, will normalize the *returns*
        by subtracting the running mean and dividing by the running standard deviation.
        Can be detrimental to performance! See TODO in process_fn.
    :param deterministic_eval: if True, will use deterministic action (the dist's mode)
        instead of stochastic one during evaluation. Does not affect training.
    :param observation_space: Env's observation space.
    :param action_scaling: if True, scale the action from [-1, 1] to the range
        of action_space. Only used if the action_space is continuous.
    :param action_bound_method: method to bound action to range [-1, 1].
        Only used if the action_space is continuous.
    :param lr_scheduler: if not None, will be called in `policy.update()`.

    .. seealso::

        Please refer to :class:`~tianshou.policy.BasePolicy` for more detailed explanation.
    """

    def __init__(
        self,
        actor: torch.nn.Module,
        optim: torch.optim.Optimizer,
        dist_fn: Type[torch.distributions.Distribution],
        action_space: gym.Space,
        obj = 'fkl',
        eps_clip: float = 0.2,
        dual_clip: Union[float, None] = None,
        ent_coef: float = 0.01,
        max_grad_norm: Union[float, None] = None,
        max_batchsize: int = 256,
        discount_factor: float = 0.99,
        agent_distribution_bandwidth: float = 0.3,
        expert_distribution_bandwidth: float = 0.15,
        num_ppo_updates: int = 20,
        # TODO: rename to return_normalization?
        reward_normalization: bool = False,
        _deterministic_eval: bool = False,
        observation_space: Union[gym.Space,  None] = None,
        # TODO: why change the default from the base?
        action_scaling: bool = True,
        **kwargs: Any,
    ) -> None:
        super().__init__(
            action_space=action_space,
            observation_space=observation_space,
            action_scaling=action_scaling,
            **kwargs,
        )
        if action_scaling and not np.isclose(actor.max_action, 1.0):
            warnings.warn(
                "action_scaling and action_bound_method are only intended"
                "to deal with unbounded model action space, but find actor model"
                f"bound action space with max_action={actor.max_action}."
                "Consider using unbounded=True option of the actor model,"
                "or set action_scaling to False and action_bound_method to None.",
            )
        self.actor = actor
        self.optim = optim
        self.dist_fn = dist_fn
        self.eps_clip = eps_clip
        self.dual_clip = dual_clip
        self.ent_coef = ent_coef
        self.max_grad_norm = max_grad_norm
        self.max_batchsize = max_batchsize
        assert 0.0 <= discount_factor <= 1.0, "discount factor should be in [0, 1]"
        self.gamma = discount_factor
        self.rew_norm = reward_normalization
        self.ret_rms = RunningMeanStd()
        self._eps = 1e-8
        self._deterministic_eval = _deterministic_eval
        self.obj = obj
        self.reward = FPGReward(obj)
        self.agent_distribution_bandwidth = agent_distribution_bandwidth
        self.expert_distribution_bandwidth = expert_distribution_bandwidth
        self.num_ppo_updates = num_ppo_updates

    def process_fn(
        self,
        batch: Batch,
        buffer: ReplayBuffer,
        indices: np.ndarray,
    ) -> Batch:
        r"""Compute the discounted returns (Monte Carlo estimates) for each transition.

        They are added to the batch under the field `returns`.
        Note: this function will modify the input batch!

        .. math::
            G_t = \sum_{i=t}^T \gamma^{i-t}r_i

        where :math:`T` is the terminal time step, :math:`\gamma` is the
        discount factor, :math:`\gamma \in [0, 1]`.

        :param batch: a data batch which contains several episodes of data in
            sequential order. Mind that the end of each finished episode of batch
            should be marked by done flag, unfinished (or collecting) episodes will be
            recognized by buffer.unfinished_index().
        :param buffer: the corresponding replay buffer.
        :param numpy.ndarray indices: tell batch's location in buffer, batch is equal
            to buffer[indices].
        """
        # Fill in the rewards
        # Get all the desired goals - with obs_indices
        desired_goals = batch.obs["desired_goal"]
        unique_desired_goals = np.unique(desired_goals, axis=0)

        for goal in unique_desired_goals:
            # Get all the indices of the desired goals
            goal_indices = np.where(np.all(desired_goals == goal, axis=1))[0]
            # Get the achieved goals
            achieved_goals = batch.obs["achieved_goal"][goal_indices]

            agent_density = neighbors.KernelDensity(bandwidth=self.agent_distribution_bandwidth, kernel='gaussian')
            agent_density.fit(achieved_goals)

            expert_density = neighbors.KernelDensity(bandwidth=self.expert_distribution_bandwidth, kernel='gaussian')
            expert_density.fit(desired_goals)

            self.reward.update(agent_density, expert_density)
            # Get the rewards
            rewards = self.reward.get_scalar_reward(achieved_goals)
            # Fill in the rewards
            batch.rew[goal_indices] = rewards


        v_s_ = np.full(indices.shape, self.ret_rms.mean)
        # gae_lambda = 1.0 means we use Monte Carlo estimate
        unnormalized_returns, _ = self.compute_episodic_return(
            batch,
            buffer,
            indices,
            v_s_=v_s_,
            gamma=self.gamma,
            gae_lambda=1.0,
        )
        # TODO: overridden in A2C, where mean is not subtracted. Subtracting mean
        #  can be very detrimental! It also has no theoretical grounding.
        #  This should be addressed soon!
        if self.rew_norm:
            batch.returns = (unnormalized_returns - self.ret_rms.mean)
            self.ret_rms.update(unnormalized_returns)
        else:
            batch.returns = unnormalized_returns

        batch.act = to_torch(batch.act, torch.float32, self.actor.device)
        logp_old = []
        with torch.no_grad():
            for minibatch in batch.split(self.max_batchsize, shuffle=False, merge_last=True):
                logp_old.append(self(minibatch).dist.log_prob(minibatch.act))
            batch.logp_old = torch.cat(logp_old, dim=0).flatten()
        return batch

    def forward(
        self,
        batch: Batch,
        state: Union[dict, Batch, np.ndarray, None]= None,
        **kwargs: Any,
    ) -> Batch:
        """Compute action over the given batch data by applying the actor.

        Will sample from the dist_fn, if appropriate.
        Returns a new object representing the processed batch data
        (contrary to other methods that modify the input batch inplace).

        .. seealso::

            Please refer to :meth:`~tianshou.policy.BasePolicy.forward` for
            more detailed explanation.
        """
        # TODO - ALGO: marked for algorithm refactoring
        action_dist_input_BD, hidden_BH = self.actor(batch.obs, state=state, info=batch.info)
        # in the case that self.action_type == "discrete", the dist should always be Categorical, and D=A
        # therefore action_dist_input_BD is equivalent to logits_BA
        # If discrete, dist_fn will typically map loc, scale to a distribution (usually a Gaussian)
        # the action_dist_input_BD in that case is a tuple of loc_B, scale_B and needs to be unpacked
        dist = self.dist_fn(action_dist_input_BD)

        act_B = (
            dist.mode
            if self.deterministic_eval and not self.is_within_training_step
            else dist.sample()
        )
        # act is of dimension BA in continuous case and of dimension B in discrete
        result = Batch(logits=action_dist_input_BD, act=act_B, state=hidden_BH, dist=dist)
        return result

    # TODO: why does mypy complain?
    def learn(  # type: ignore
        self,
        batch: Batch,
        batch_size: Union[int, None],
        repeat: int,
        *args: Any,
        **kwargs: Any,
    ) -> Dict[str, Any]:
        losses, clip_losses, ent_losses = [], [], []
        gradient_steps = 0        
        split_batch_size = batch_size or -1
        for _ in range(repeat):
            for minibatch in batch.split(split_batch_size, merge_last=True):
                gradient_steps += 1
                dist = self(minibatch).dist
                rets = to_torch(minibatch.returns, torch.float, self.actor.device)
                ratios = (dist.log_prob(minibatch.act) - minibatch.logp_old).exp().float()
                ratios = ratios.reshape(ratios.size(0), -1).transpose(0, 1)
                surr1 = ratios * rets
                surr2 = ratios.clamp(1.0 - self.eps_clip, 1.0 + self.eps_clip) * rets
                if self.dual_clip:
                    clip1 = torch.min(surr1, surr2)
                    clip2 = torch.max(clip1, self.dual_clip * rets)
                    clip_loss = -torch.where(rets < 0, clip2, clip1).mean()
                else:
                    clip_loss = -torch.min(surr1, surr2).mean()
    
                # calculate regularization and overall loss
                ent_loss = dist.entropy().mean()
                loss = clip_loss - self.ent_coef * ent_loss
                self.optim.zero_grad()
                result = self(minibatch)
                dist = result.dist
                loss.backward()
                if self.max_grad_norm:  # clip large gradient
                    nn.utils.clip_grad_norm_(
                        self.actor.parameters(),
                        max_norm=self.max_grad_norm,
                    )
                self.optim.step()
                losses.append(loss.item())
                clip_losses.append(clip_loss.item())
                ent_losses.append(ent_loss.item())

        summary = dict(
            loss = losses,
            clip_loss = clip_losses,
            ent_loss = ent_losses,
            gradient_steps = gradient_steps
        )

        return summary